﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/codecommit/model/Location.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class PostCommentForComparedCommitRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API PostCommentForComparedCommitRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PostCommentForComparedCommit"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the repository where you want to post a comment on the comparison
   * between commits.</p>
   */
  inline const Aws::String& GetRepositoryName() const { return m_repositoryName; }
  inline bool RepositoryNameHasBeenSet() const { return m_repositoryNameHasBeenSet; }
  template <typename RepositoryNameT = Aws::String>
  void SetRepositoryName(RepositoryNameT&& value) {
    m_repositoryNameHasBeenSet = true;
    m_repositoryName = std::forward<RepositoryNameT>(value);
  }
  template <typename RepositoryNameT = Aws::String>
  PostCommentForComparedCommitRequest& WithRepositoryName(RepositoryNameT&& value) {
    SetRepositoryName(std::forward<RepositoryNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>To establish the directionality of the comparison, the full commit ID of the
   * before commit. Required for commenting on any commit unless that commit is the
   * initial commit.</p>
   */
  inline const Aws::String& GetBeforeCommitId() const { return m_beforeCommitId; }
  inline bool BeforeCommitIdHasBeenSet() const { return m_beforeCommitIdHasBeenSet; }
  template <typename BeforeCommitIdT = Aws::String>
  void SetBeforeCommitId(BeforeCommitIdT&& value) {
    m_beforeCommitIdHasBeenSet = true;
    m_beforeCommitId = std::forward<BeforeCommitIdT>(value);
  }
  template <typename BeforeCommitIdT = Aws::String>
  PostCommentForComparedCommitRequest& WithBeforeCommitId(BeforeCommitIdT&& value) {
    SetBeforeCommitId(std::forward<BeforeCommitIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>To establish the directionality of the comparison, the full commit ID of the
   * after commit.</p>
   */
  inline const Aws::String& GetAfterCommitId() const { return m_afterCommitId; }
  inline bool AfterCommitIdHasBeenSet() const { return m_afterCommitIdHasBeenSet; }
  template <typename AfterCommitIdT = Aws::String>
  void SetAfterCommitId(AfterCommitIdT&& value) {
    m_afterCommitIdHasBeenSet = true;
    m_afterCommitId = std::forward<AfterCommitIdT>(value);
  }
  template <typename AfterCommitIdT = Aws::String>
  PostCommentForComparedCommitRequest& WithAfterCommitId(AfterCommitIdT&& value) {
    SetAfterCommitId(std::forward<AfterCommitIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The location of the comparison where you want to comment.</p>
   */
  inline const Location& GetLocation() const { return m_location; }
  inline bool LocationHasBeenSet() const { return m_locationHasBeenSet; }
  template <typename LocationT = Location>
  void SetLocation(LocationT&& value) {
    m_locationHasBeenSet = true;
    m_location = std::forward<LocationT>(value);
  }
  template <typename LocationT = Location>
  PostCommentForComparedCommitRequest& WithLocation(LocationT&& value) {
    SetLocation(std::forward<LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content of the comment you want to make.</p>
   */
  inline const Aws::String& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = Aws::String>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = Aws::String>
  PostCommentForComparedCommitRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, client-generated idempotency token that, when provided in a
   * request, ensures the request cannot be repeated with a changed parameter. If a
   * request is received with the same parameters and a token is included, the
   * request returns information about the initial request that used that token.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  PostCommentForComparedCommitRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_repositoryName;
  bool m_repositoryNameHasBeenSet = false;

  Aws::String m_beforeCommitId;
  bool m_beforeCommitIdHasBeenSet = false;

  Aws::String m_afterCommitId;
  bool m_afterCommitIdHasBeenSet = false;

  Location m_location;
  bool m_locationHasBeenSet = false;

  Aws::String m_content;
  bool m_contentHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
