﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/config/ConfigServiceRequest.h>
#include <aws/config/ConfigService_EXPORTS.h>
#include <aws/config/model/RemediationExceptionResourceKey.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace ConfigService {
namespace Model {

/**
 */
class PutRemediationExceptionsRequest : public ConfigServiceRequest {
 public:
  AWS_CONFIGSERVICE_API PutRemediationExceptionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutRemediationExceptions"; }

  AWS_CONFIGSERVICE_API Aws::String SerializePayload() const override;

  AWS_CONFIGSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the Config rule for which you want to create remediation
   * exception.</p>
   */
  inline const Aws::String& GetConfigRuleName() const { return m_configRuleName; }
  inline bool ConfigRuleNameHasBeenSet() const { return m_configRuleNameHasBeenSet; }
  template <typename ConfigRuleNameT = Aws::String>
  void SetConfigRuleName(ConfigRuleNameT&& value) {
    m_configRuleNameHasBeenSet = true;
    m_configRuleName = std::forward<ConfigRuleNameT>(value);
  }
  template <typename ConfigRuleNameT = Aws::String>
  PutRemediationExceptionsRequest& WithConfigRuleName(ConfigRuleNameT&& value) {
    SetConfigRuleName(std::forward<ConfigRuleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An exception list of resource exception keys to be processed with the current
   * request. Config adds exception for each resource key. For example, Config adds 3
   * exceptions for 3 resource keys. </p>
   */
  inline const Aws::Vector<RemediationExceptionResourceKey>& GetResourceKeys() const { return m_resourceKeys; }
  inline bool ResourceKeysHasBeenSet() const { return m_resourceKeysHasBeenSet; }
  template <typename ResourceKeysT = Aws::Vector<RemediationExceptionResourceKey>>
  void SetResourceKeys(ResourceKeysT&& value) {
    m_resourceKeysHasBeenSet = true;
    m_resourceKeys = std::forward<ResourceKeysT>(value);
  }
  template <typename ResourceKeysT = Aws::Vector<RemediationExceptionResourceKey>>
  PutRemediationExceptionsRequest& WithResourceKeys(ResourceKeysT&& value) {
    SetResourceKeys(std::forward<ResourceKeysT>(value));
    return *this;
  }
  template <typename ResourceKeysT = RemediationExceptionResourceKey>
  PutRemediationExceptionsRequest& AddResourceKeys(ResourceKeysT&& value) {
    m_resourceKeysHasBeenSet = true;
    m_resourceKeys.emplace_back(std::forward<ResourceKeysT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The message contains an explanation of the exception.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  PutRemediationExceptionsRequest& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The exception is automatically deleted after the expiration date.</p>
   */
  inline const Aws::Utils::DateTime& GetExpirationTime() const { return m_expirationTime; }
  inline bool ExpirationTimeHasBeenSet() const { return m_expirationTimeHasBeenSet; }
  template <typename ExpirationTimeT = Aws::Utils::DateTime>
  void SetExpirationTime(ExpirationTimeT&& value) {
    m_expirationTimeHasBeenSet = true;
    m_expirationTime = std::forward<ExpirationTimeT>(value);
  }
  template <typename ExpirationTimeT = Aws::Utils::DateTime>
  PutRemediationExceptionsRequest& WithExpirationTime(ExpirationTimeT&& value) {
    SetExpirationTime(std::forward<ExpirationTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_configRuleName;
  bool m_configRuleNameHasBeenSet = false;

  Aws::Vector<RemediationExceptionResourceKey> m_resourceKeys;
  bool m_resourceKeysHasBeenSet = false;

  Aws::String m_message;
  bool m_messageHasBeenSet = false;

  Aws::Utils::DateTime m_expirationTime{};
  bool m_expirationTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ConfigService
}  // namespace Aws
